#ifndef UTILS_H
#define UTILS_H

#include <sys/resource.h>
#include <sys/time.h>

#include <chrono>
#include <cmath>
#include <functional>
#include <random>

#define GOOGLE_STRIP_LOG 0  // 1 to remove all INFO and VLOG from this file
#include <glog/logging.h>

// allows every file that #include "utils.hpp" to see this bool
#include <gflags/gflags_declare.h>
DECLARE_bool(benchmark);

/************
 *  Timing  *
 ************/

typedef std::chrono::high_resolution_clock::time_point TimeVar;
constexpr auto timeNow = std::chrono::high_resolution_clock::now;
inline const auto duration = [](const auto t) {
  return std::chrono::duration_cast<std::chrono::milliseconds>(t).count();
};

/********************
 *  Random numbers  *
 ********************/

static std::random_device rd;
static std::mt19937 seed(rd());

inline auto random_proba =
    std::bind(std::uniform_real_distribution<float>(0, 1), ref(seed));

/**********
 *  Misc  *
 **********/

template<typename Floating>
constexpr bool almostEqual(Floating a, Floating b,
                           Floating tolerance = 0.0001) {
  return std::abs(a - b) < tolerance;
}

inline void hash_combine(std::size_t&) {}

template<typename T, typename... Rest>
inline void hash_combine(std::size_t& seedValue, const T& v, Rest... rest) {
  std::hash<T> hasher;
  seed ^= hasher(v) + 0x9e3779b9 + (seedValue << 6) + (seedValue >> 2);
  hash_combine(seed, rest...);
}

inline long getMaxMemoryUsed() {
  struct rusage usage;
  getrusage(RUSAGE_SELF, &usage);
  return usage.ru_maxrss / 1000;
}

#endif
