#ifndef UTILS_H
#define UTILS_H

#include <chrono>
#include <cmath>
#include <functional>
#include <random>
#include <sys/time.h>
#include <sys/resource.h>

/************
*  Timing  *
************/

typedef std::chrono::high_resolution_clock::time_point TimeVar;
constexpr auto timeNow = std::chrono::high_resolution_clock::now;
inline const auto duration = [](const auto t){
  return std::chrono::duration_cast<std::chrono::milliseconds>(t).count();
};

inline auto benchmark(std::function<void()> f, size_t n_repeat = 1) {
  const TimeVar t1 = timeNow();
  for (size_t i = 0; i < n_repeat; ++i)
    f();
  const TimeVar t2 = timeNow();
  return static_cast<size_t>(duration(t2 - t1)) / n_repeat;
}

/**********
*  Misc  *
**********/

static std::random_device rd;
static std::mt19937 gen(rd());

// approximates log(k!) using stirling approximation
constexpr size_t logFactorial(size_t k) {
  double k_d = static_cast<double>(k);
  return static_cast<size_t>((k_d * log(k_d)) - k_d + (log(2*M_PI*k_d) / 2));
}

template <typename Floating>
constexpr bool almostEqual(Floating a, Floating b,
                           Floating tolerance = 0.0001) {
  return std::abs(a - b) < tolerance;
}

inline long getMaxMemoryUsed() {
  struct rusage usage;
  getrusage(RUSAGE_SELF, &usage);
  return usage.ru_maxrss / 1000;
}

#endif
