#include "point.hpp"

#include <cassert>
#include <cmath>

constexpr long Rayon_Terre = 6'371'000;  // en mètres

using namespace std;

Point::Point() : Point(0, 0) {}
Point::Point(double lati, double longi) : latitude(lati), longitude(longi) {}

double Point::distance(const Point& point) const {
  double s1 = sin((point.latitude - latitude) / 2);
  double s2 = sin((point.longitude - longitude) / 2);
  return 2 * Rayon_Terre
         * asin(sqrt(s1 * s1 + cos(latitude) * cos(point.latitude) * s2 * s2));
}

Point Point::getPointMilieu(const Point& point) const {
  double avgLat = (latitude + point.latitude) / 2.0;
  double avgLong = (longitude + point.longitude) / 2.0;
  return Point(avgLat, avgLong);
}

Point& Point::operator+=(const Point& p) {
  latitude += p.latitude;
  longitude += p.longitude;
  return *this;
}

const double& Point::operator[](size_t dim) const {
  if (dim == 0)
    return latitude;
  else if (dim == 1)
    return longitude;
  throw runtime_error("Invalid Point index");
}

double& Point::operator[](size_t dim) {
  if (dim == 0)
    return latitude;
  else if (dim == 1)
    return longitude;
  throw runtime_error("Invalid Point index");
}

istream& operator>>(istream& is, Point& point) {
  char po, vir, pf;
  is >> po;
  if (is) {
    is >> point.latitude >> vir >> point.longitude >> pf;
    assert(po == '(');
    assert(vir == ',');
    assert(pf == ')');
    point.longitude *= M_PI / 180.0;
    point.latitude *= M_PI / 180.0;
  }
  return is;
}

ostream& operator<<(ostream& os, const Point& point) {
  os << "(" << (point.latitude * 180.0 / M_PI) << ","
     << (point.longitude * 180.0 / M_PI) << ")";
  return os;
}
