#include <iostream>
#include <fstream>

#include "types.hpp"
#include "Ve.hpp"
#include "Graph.hpp"
#include "cxxopts.hpp"
#include "algo.hpp"
#include "ibp.hpp"

using namespace std;

vector<vector<Ve>> loadRequests(const string& reqFilepath,
                                [[maybe_unused]] size_t nodeCount) {
  TimeScope("Loading requests");
  vector<vector<Ve>> requests;
  vector<Ve> request;

  ifstream inputData(reqFilepath);
  string line;
  while(getline(inputData, line)) {
    if(line == "===") {
      requests.push_back(request);
      request.clear();
    } else {
      stringstream ss(line);

      minutes startTime;
      nodeId start;
      nodeId dest;
      meters autonomy;
      ss >> startTime >> start >> dest >> autonomy;

      assert(start < nodeCount && dest < nodeCount);
      request.push_back(Ve(size(request), start, dest, autonomy, startTime));
    }
  }

  return requests;
}

globalSolution launchSolver(const string& solver, const Graph& graph,
                            const vector<Ve>& request) {
  TimeScope("Solving requests");
  graph.reinitState();
  graph.reinitBookings();

  if(solver == "nocoop")
    return AStarGrouped<TypeShuffle::asIs, TypeReserv::noCoop>(graph, request);
  else if(solver == "coopAsIs")
    return AStarGrouped<TypeShuffle::asIs>(graph, request);
  else if(solver == "coopStartFirst")
    return AStarGrouped<TypeShuffle::startFirst>(graph, request);
  else if(solver == "coopLogFact")
    return AStarGrouped<TypeShuffle::logFact>(graph, request);
  else if(solver == "coopCascade")
    return AStarGrouped<TypeShuffle::cascade>(graph, request);
  else if(solver == "permutations")
    return AStarGrouped<TypeShuffle::permutations>(graph, request);
  else if(solver == "optimal")
    return runOptimal(graph, request);

  cerr << "Invalid solver name" << endl;
  exit(EXIT_FAILURE);
}

void printResults(const Graph& graph, const vector<Ve>& request,
                  const globalSolution& results) {
  TimeScope("Compute and print metrics");

  minutes movingTimes = 0;
  minutes reloadingTimes = 0;
  minutes waitingTimes = 0;
  meters sumDistances = 0;
  for(size_t i = 0; i < size(results.first); ++i) {
    const auto& [route, routeTime, routeLength] = results.first[i];
    const Ve& currentVe = request[i];

    minutes currentMovingTime = 0;
    minutes currentReloadingTime = 0;
    for(size_t j = 1; j < size(route); ++j) {
      const auto& [movingTime, reloadingTime]
        = currentVe.computeHopCost(graph.getEdgeDistance(route[j-1], route[j]));
      movingTimes += movingTime;
      currentMovingTime += movingTime;

      // Reloading only if before last node
      if(j < size(route) - 1) {
        reloadingTimes += reloadingTime;
        currentReloadingTime += reloadingTime;
      }
    }

    waitingTimes += routeTime - (currentMovingTime + currentReloadingTime);
    sumDistances += routeLength;
  }

  // results.second: penalty
  cout << movingTimes << "\t" << reloadingTimes << "\t" << waitingTimes << "\t"
    << sumDistances << "\t" << results.second << "\t";
}

int main(int argc, char* argv[]) {
  ibp::BeginProfiler();

  cxxopts::ParseResult args;
  {
    TimeScope("Parse arguments");
    cxxopts::Options options("veCoop",
        "A program to compute multi-EVs global plans");

    options.add_options()
      ("f,filename", "Path to the filename of the input graph", cxxopts::value<string>())
      ("r,requests", "Path to the filename of the input requests", cxxopts::value<string>())
      ("s,solver", "Name of the solver to use (nocoop, coop[AsIs|StartFirst|LogFact|Cascade], optimal)", cxxopts::value<string>())
      ("g,generate", "Generate a file of EV requests", cxxopts::value<size_t>())
      ("n,numEVs", "Number of EVs per generated request", cxxopts::value<size_t>())
      ("m,mode", "Mode of generation (0: uniform; 1: cluster)", cxxopts::value<int>())
      ("h,help", "Print usage") ;
    args = options.parse(argc, argv);

    if (args.count("help") || argc < 2) {
      cout << options.help() << endl;
      return EXIT_SUCCESS;
    }
  }

  const string filename = args["filename"].as<string>();
  const Graph graph(filename);

  if(args.count("generate")) {
    Graph::GenMode mode;
    switch(args["mode"].as<int>()) {
      case 0: mode = Graph::Uniform; break;
      case 1: mode = Graph::Cluster; break;
      default: return EXIT_FAILURE;
    };

    const size_t numReqs = args["generate"].as<size_t>();
    const size_t numEVs = args["numEVs"].as<size_t>();

    graph.generateRequests(mode, numReqs, numEVs);
    return EXIT_SUCCESS;
  }

  const string solver = args["solver"].as<string>();
  const string reqFilepath = args["requests"].as<string>();

  const vector<vector<Ve>> requests = loadRequests(reqFilepath, graph.getNodeCount());
  for(const auto& currRequest : requests) {
    globalSolution results = launchSolver(solver, graph, currRequest);
    printResults(graph, currRequest, results);
    cout << endl;
  }

#ifdef PROFILER
  ibp::StopAndPrintProfiler();
#endif
  return EXIT_SUCCESS;
}
