#pragma once

#include <unordered_map>
#include <unordered_set>
#include <queue>
#include <numeric>

#include "types.hpp"
#include "Graph.hpp"
#include "Ve.hpp"
#include "utils.hpp"

enum class TypeReserv  { coop, noCoop, noReserve };
enum class TypeShuffle { asIs, startFirst, logFact, cascade, permutations };

std::vector<unitarySolution> calculerN4(const Graph&, const std::vector<Ve>&);
void SimulateWaiting(const Graph& nodeGraph, const std::vector<Ve>& veList,
                     std::vector<unitarySolution>& results);
globalSolution runOptimal(const Graph& nodeGraph, const std::vector<Ve>& vehicleList);

template<TypeReserv type>
unitarySolution
extractPath(const Graph& graph, nodeId startNode, nodeId endNode,
            std::unordered_map<nodeId, std::pair<minutes, minutes>> candidates,
            minutes startTime, const Ve& vehicle)
{
  std::vector<nodeId> buildedPath;
  minutes travelTime = 0;
  meters totaldistance = 0;

  const auto& backtrack = graph.backtrackPath(startNode, endNode);
  buildedPath = backtrack.first;

  if constexpr(type == TypeReserv::coop)
    travelTime = backtrack.second-startTime;
  else
    travelTime = startTime; // Temporary init

  for(size_t i = 1; i < size(buildedPath); ++i)
  {
    const meters edgeDistance = graph.getEdgeDistance(buildedPath[i-1], buildedPath[i]);
    std::pair<minutes, minutes> hopCost;

    // Adding travelDistance
    totaldistance += edgeDistance;

    // If no coop adding travel time on the fly
    if constexpr(type == TypeReserv::noCoop || type == TypeReserv::noReserve)
    {
      hopCost = vehicle.computeHopCost(edgeDistance);
      travelTime += hopCost.first;
    }

    if(i != size(buildedPath) - 1)
    {
      if constexpr(type == TypeReserv::coop)
      {
        const auto& selectedBooking = candidates[buildedPath[i]];
        graph.bookReservation(buildedPath[i], vehicle.getId(),
            selectedBooking.first, selectedBooking.second);
      }
      else
      {
        if constexpr(type == TypeReserv::noCoop)
          // If no coop, prebook without taking account for previous bookings
          graph.bookReservation(buildedPath[i], vehicle.getId(),
              travelTime, travelTime+hopCost.second);
        travelTime += hopCost.second;
      }
    }
    // If no coop, last insertion needed to keep track for arrival time durring simulation
    else if constexpr(type == TypeReserv::noCoop)
      graph.bookReservation(buildedPath[i], vehicle.getId(), travelTime, travelTime);
  }

  // Once travel time compute finished, removing start time added at start
  if constexpr(type == TypeReserv::noCoop || type == TypeReserv::noReserve)
    travelTime -= startTime;

  // Reinit state before leaving
  graph.reinitState();

  return {buildedPath, travelTime, totaldistance};
}

template<TypeReserv type>
unitarySolution AStar(const Graph& nodeGraph, const Ve& vehicle)
{
  const nodeId startNode  = vehicle.getStart();
  const nodeId endNode    = vehicle.getDest();
  const minutes startTime = vehicle.getStartTime();

  const auto nodeComparator = [](const std::pair<nodeId, meters>& a,
                                 const std::pair<nodeId, meters>& b) {
    return a.second > b.second;
  };

  std::unordered_map<nodeId, std::pair<minutes, minutes>> bookingCandidates;
  std::priority_queue<std::pair<nodeId, meters>,
                      std::vector<std::pair<nodeId, meters>>,
                      decltype(nodeComparator)> nodeQueue{nodeComparator};

  // init graph
  nodeGraph.updateNode(startNode, startTime, startNode);
  const meters tempDist = nodeGraph.getHeuristicDistance(startNode,endNode);
  nodeQueue.emplace(startNode, startTime + vehicle.computeTravelUnbounded(tempDist));

  bool pathFound = false;
  while(!nodeQueue.empty())
  {
    const nodeId currentNode = nodeQueue.top().first;
    const minutes currentTime = nodeGraph.getCurrentTime(currentNode);
    nodeQueue.pop();

    if(currentNode == endNode)
    {
      pathFound = true;
      break;
    }

    if(!nodeGraph.isVisited(currentNode))
    {
      nodeGraph.setVisited(currentNode);
      for(const auto& [succNode, dist] : nodeGraph.getSuccessors(currentNode))
      {
        if(dist < vehicle.getAutonomy())
        {
          const auto& [travelTime, chargeTime] = vehicle.computeHopCost(dist);
          const minutes arrivalTime = currentTime + travelTime;

          std::pair<minutes,minutes> waitPeriod;

          if constexpr(type == TypeReserv::coop)
            waitPeriod = nodeGraph.getNextSpace(succNode, arrivalTime, chargeTime);
          else
            // If not in coop mode, load immediately
            waitPeriod = {arrivalTime, arrivalTime + chargeTime};

          if(succNode != endNode)
          {
            if(!nodeGraph.isInserted(succNode) || nodeGraph.getCurrentTime(succNode) > waitPeriod.second)
            {
              if constexpr(type == TypeReserv::coop)
                bookingCandidates[succNode] = waitPeriod;

              nodeGraph.updateNode(succNode, waitPeriod.second, currentNode);
              const meters flyDist = nodeGraph.getHeuristicDistance(succNode, endNode);
              const minutes newMinutes = waitPeriod.second + vehicle.computeTravelUnbounded(flyDist);
              nodeQueue.emplace(succNode, newMinutes);
              nodeGraph.unsetVisited(succNode);
              nodeGraph.setInserted(succNode);
            }
          }
          else
          {
            // Take arrival time because no reload/no wait
            if(!nodeGraph.isInserted(succNode) || nodeGraph.getCurrentTime(succNode) > arrivalTime)
            {
              nodeGraph.updateNode(succNode, arrivalTime, currentNode);
              nodeQueue.emplace(succNode, arrivalTime);
              nodeGraph.setInserted(succNode);
            }
          }
        }
      }
    }
  }

  if(pathFound)
    return extractPath<type>(nodeGraph, startNode, endNode, bookingCandidates, startTime, vehicle);
  nodeGraph.reinitState();
  return {{}, 0, 0};
}

template<TypeReserv type = TypeReserv::coop>
globalSolution solvePermutation(const Graph& graph, std::vector<size_t> ids,
                                const std::vector<Ve>& veList, std::vector<minutes> refTimes)
{
  std::vector<unitarySolution> results(size(veList));
  double penality = 0;
  for(const auto& currentVehicleId : ids)
  {
    results[currentVehicleId] = AStar<type>(graph, veList[currentVehicleId]);

    // If no coop, we need to simulate before fetching the difference and results
    if constexpr(type != TypeReserv::noCoop)
    {
      const double difference
        = static_cast<double>(std::get<1>(results[currentVehicleId]) - refTimes[currentVehicleId]);
      penality += difference * difference;
    }
  }

  //Simulate collisions and update values
  if constexpr(type == TypeReserv::noCoop)
  {
    SimulateWaiting(graph, veList, results);

    //Calculate global score
    for(const auto& currentVehicleId : ids)
    {
      const double difference
        = static_cast<double>(std::get<1>(results[currentVehicleId]) - refTimes[currentVehicleId]);
      penality += difference * difference;
    }
  }

  double tmpVehicleListSize = static_cast<double>(size(veList));
  penality /= tmpVehicleListSize;
  graph.reinitBookings();

  return {results, penality};
}

template<TypeShuffle shuffle, TypeReserv type = TypeReserv::coop>
globalSolution AStarGrouped(const Graph& nodeGraph, const std::vector<Ve>& vehicleList)
{
  std::vector<minutes> sequencesScore;
  std::vector<size_t> ids(size(vehicleList));
  std::iota(begin(ids), end(ids), 0);

  // Computing reference for each vehicle, no coop, no wait
  std::vector<minutes> refTimes;
  for(const auto& currentVe : vehicleList)
    refTimes.push_back(std::get<1>(AStar<TypeReserv::noReserve>(nodeGraph, currentVe)));

  std::vector<globalSolution> resultPool;

  if constexpr(shuffle == TypeShuffle::permutations)
  {
    do
      resultPool.push_back(solvePermutation(nodeGraph, ids, vehicleList, refTimes));
    while (next_permutation(begin(ids), end(ids)));
    return *std::min_element(begin(resultPool), end(resultPool), [](const auto& a, const auto& b) {
      return a.second < b.second;
    });
  }
  else if constexpr(shuffle == TypeShuffle::cascade)
  {
    std::vector<std::vector<size_t>> permutations;
    size_t previousOrigin = 0;
    size_t previousDestination = 0;

    for(size_t origin = 0; origin < size(ids) - 1; ++origin)
    {
      for(size_t destination = origin + 1; destination < size(ids); ++destination)
      {
        resultPool.push_back(solvePermutation(nodeGraph, ids, vehicleList, refTimes));

        // compute next cascade permutation
        size_t tmp = ids[previousOrigin];
        ids[previousOrigin] = ids[previousDestination];
        ids[previousDestination] = tmp;

        previousOrigin = origin;
        previousDestination = destination;

        tmp = ids[origin];
        ids[origin] = ids[destination];
        ids[destination] = tmp;
      }
    }

    // Last permutation:
    resultPool.push_back(solvePermutation(nodeGraph, ids, vehicleList, refTimes));

    return *std::min_element(begin(resultPool), end(resultPool), [](const auto& a, const auto& b) {
      return a.second < b.second;
    });
  }
  else if constexpr(shuffle == TypeShuffle::logFact)
  {
    for(size_t i = 0, i_end = logFactorial(size(vehicleList)) + 1; i < i_end; ++i) {
      std::shuffle(begin(ids), end(ids), gen);
      resultPool.push_back(solvePermutation(nodeGraph, ids, vehicleList, refTimes));
    }

    return *std::min_element(begin(resultPool), end(resultPool), [](const auto& a, const auto& b) {
      return a.second < b.second;
    });
  }
  else // asIs or startFirst
  {
    if constexpr(shuffle == TypeShuffle::startFirst)
      std::sort(ids.begin(), ids.end(), [vehicleList](auto a, auto b) {
          return vehicleList[a].getStartTime() < vehicleList[b].getStartTime();
      });

    return solvePermutation(nodeGraph, ids, vehicleList, refTimes);
  }
}
