#pragma once

#include <vector>
#include <string>

#include "types.hpp"
#include "point.hpp"
#include "matrix.hpp"

class Graph
{
  private:
    size_t nodeCount = 0;
    size_t edgeCount = 0;

    std::vector<std::string> nodesNames{};
    std::vector<Point> nodesPositions{};

    std::vector<size_t> nodes{};
    std::vector<nodeId> edges{};
    std::vector<meters> weights{};
    Matrix<meters> distances{};

    // Structure used by algorithm while generating solution
    mutable std::vector<minutes> currentTime{};
    mutable std::vector<nodeId> parents{};
    mutable std::vector<bool> visited{};
    mutable std::vector<bool> inserted{};

    mutable std::vector<std::vector<std::tuple<minutes,minutes,veId>>> bookings{};
    const Matrix<meters>& getDistances();
    std::vector<nodeId> getCluster(nodeId center) const;

  public:
    Graph(const std::string&);

    std::vector<std::pair<nodeId, meters>> getSuccessors(nodeId) const;

    size_t getNodeCount() const;

    meters getEdgeDistance(nodeId, nodeId) const;
    meters getFlyDistance(nodeId, nodeId) const;
    meters getHeuristicDistance(nodeId, nodeId) const;

    void updateNode(nodeId,minutes,nodeId) const;

    bool isInserted(nodeId) const;
    void setInserted(nodeId) const;

    bool isVisited(nodeId) const;
    void setVisited(nodeId) const;
    void unsetVisited(nodeId) const;

    minutes getCurrentTime(nodeId) const;

    std::pair<minutes,minutes> getNextSpace(nodeId, minutes, minutes) const;
    void bookReservation(nodeId,veId,minutes, minutes) const;

    std::pair<std::vector<nodeId>,minutes> backtrackPath(nodeId, nodeId) const;

    enum GenMode { Uniform, Cluster };
    void generateRequests(GenMode mode, size_t numReqs, size_t numEVs,
                          meters minRange      = 100'000u,
                          meters maxRange      = 550'000u,
                          minutes maxStartTime = 60u) const;

    std::vector<std::pair<veId, minutes>> fixNode(nodeId) const;
    void driftBooking(nodeId, veId, minutes) const;
    std::pair<minutes, minutes> getBooking(nodeId, veId) const;

    void reinitBookings() const;
    void reinitState() const;
};
