#include <iostream>
#include <fstream>
#include <tuple>
#include <algorithm>
#include <map>
#include <random>
#include <cmath>
#include <filesystem>
#include <iostream>

#include "Graph.hpp"
#include "algo.hpp"
#include "utils.hpp"
#include "ibp.hpp"

using namespace std;

Graph::Graph(const string& inputPath)
{
  TimeScope("Loading graph");

  ifstream inputData(inputPath);
  vector<tuple<size_t, string, Point>> loadedNodes;
  vector<tuple<nodeId, nodeId, meters>> loadedEdges;

  // Loading nodes data
  size_t nodeIndex;
  string currentNodeName;
  Point currentNodePoint;
  do
  {
    inputData >> currentNodeName;
    if(currentNodeName != "---")
    {
      inputData >> nodeIndex;
      inputData >> currentNodePoint;
      loadedNodes.push_back({nodeIndex, currentNodeName, currentNodePoint});
    }
  } while(inputData.good() && currentNodeName != "---");

  do
  {
    nodeId beginId;
    nodeId endId;
    double loadedDist;

    inputData >> beginId;
    inputData >> endId;
    inputData >> loadedDist;

    loadedEdges.push_back({beginId, endId, static_cast<meters>(loadedDist)});
  } while(inputData.good());

  nodeCount = size(loadedNodes);
  edgeCount = size(loadedEdges);

  nodesNames = vector<string>(nodeCount);
  nodesPositions = vector<Point>(nodeCount);
  nodes = vector<size_t>(nodeCount + 1);

  edges = vector<nodeId>(edgeCount);
  weights = vector<meters>(edgeCount);

  currentTime = vector<meters>(nodeCount, 0);
  parents = vector<nodeId>(nodeCount);
  visited = vector<bool>(nodeCount, false);
  inserted = vector<bool>(nodeCount, false);

  bookings = vector<vector<tuple<minutes, minutes, veId>>>(nodeCount);

  for(const auto& [index, name, pos] : loadedNodes)
  {
    nodesNames[index - 1] = name;
    nodesPositions[index - 1] = pos;
  }

  nodes[0] = 0;
  nodes[size(nodes) - 1] = size(edges) - 1;

  sort(begin(loadedEdges), end(loadedEdges));

  nodeId lastId = 0;
  size_t currentPosition = 0;
  size_t previousPosition = 0;

  for(const auto& [startNode, endNode, dist] : loadedEdges)
  {
    if(startNode - 1 != lastId)
    {
      previousPosition = lastId;
      lastId = startNode - 1;
      for(size_t i = previousPosition + 1; i < lastId + 1; ++i)
        nodes[i] = currentPosition;
    }

    edges[currentPosition] = endNode - 1;
    weights[currentPosition] = dist - 1;
    ++currentPosition;
  }

  // load floyd-warshall data in matrix if already computed on disk
  string matrixPath = inputPath;
  matrixPath.replace(matrixPath.find(".graphe"), 7, ".matrix");
  if(filesystem::exists(matrixPath)) {
    distances = Matrix<meters>(matrixPath);
  } else {
    getDistances();
    distances.exportToFile(matrixPath);
  }
}

size_t Graph::getNodeCount() const { return nodeCount; }

vector<pair<nodeId,meters>> Graph::getSuccessors(nodeId searchedNode) const
{
  vector<pair<nodeId,meters>> buildedList;
  for(size_t i = nodes[searchedNode], i_end = nodes[searchedNode + 1]; i < i_end; ++i)
    buildedList.push_back({edges[i], weights[i]});
  return buildedList;
}

meters Graph::getEdgeDistance(nodeId startId, nodeId arrivalId) const
{
  for(size_t i = nodes[startId], i_end = nodes[startId + 1]; i < i_end; i++)
    if(edges[i] == arrivalId)
      return weights[i];
  return 0;
}

meters Graph::getFlyDistance(nodeId startId, nodeId arrivalId) const
{
  // use floor to ensure admissibility
  return static_cast<meters>(floor(nodesPositions[startId].distance(nodesPositions[arrivalId])));
}

meters Graph::getHeuristicDistance(nodeId startId, nodeId arrivalId) const
{
  return distances(startId + 1, arrivalId + 1);
}

const Matrix<meters>& Graph::getDistances() {
  if(distances.getNbLines() != 0)
    return distances;

  distances = Matrix<meters>(nodeCount);
  distances.setElemAllCells(numeric_limits<meters>::max());
  for(size_t from = 1; from <= nodeCount; ++from)
  {
    distances(from, from) = 0;
    for(const auto& [to_m1, dist] : getSuccessors(from - 1))
      distances(from, to_m1 + 1) = dist;
  }

  for(size_t k = 1; k <= nodeCount; ++k)
    for(size_t i = 1; i <= nodeCount; ++i)
      for(size_t j = 1; j <= nodeCount; ++j)
        distances(i, j) = min(distances(i, j), distances(i, k) + distances(k, j));

  return distances;
}

void Graph::updateNode(nodeId concernedNode, minutes nodeTime, nodeId parent) const
{
  currentTime[concernedNode] = nodeTime;
  parents[concernedNode] = parent;
}

bool Graph::isInserted(nodeId node) const { return inserted[node]; }
void Graph::setInserted(nodeId node) const { inserted[node] = true; }
bool Graph::isVisited(nodeId node) const { return visited[node]; }
void Graph::setVisited(nodeId node) const { visited[node] = true; }
void Graph::unsetVisited(nodeId node) const { visited[node] = false; }
minutes Graph::getCurrentTime(nodeId node) const { return currentTime[node]; }
void Graph::reinitBookings() const { bookings.assign(nodeCount, {}); }

void Graph::reinitState() const
{
  currentTime.assign(nodeCount,0);
  parents.assign(nodeCount, 0);
  visited.assign(nodeCount,false);
  inserted.assign(nodeCount,false);
}

pair<minutes,minutes> Graph::getNextSpace(nodeId concernedNode,
    minutes arrivalHour, minutes loadingDuration) const
{
  const auto& selectedNodeBooking = bookings[concernedNode];
  size_t currentIndex = 0;

  // Case if it can be fitted before first
  if(size(selectedNodeBooking) == 0 || get<0>(selectedNodeBooking[currentIndex]) >= arrivalHour + loadingDuration)
    return {arrivalHour, arrivalHour + loadingDuration};

  const size_t iterLimit = selectedNodeBooking.size()-1;
  while(get<1>(selectedNodeBooking[currentIndex]) < arrivalHour && currentIndex < iterLimit)
    currentIndex++;

  if(currentIndex < iterLimit)
  {
    currentIndex--;
    while(currentIndex < iterLimit)
    {
      if(get<0>(selectedNodeBooking[currentIndex + 1]) > arrivalHour
          && loadingDuration <= get<0>(selectedNodeBooking[currentIndex + 1])
             - max(get<1>(selectedNodeBooking[currentIndex]), arrivalHour))
      {
        const auto selectedValue = max(get<1>(selectedNodeBooking[currentIndex]), arrivalHour);
        return {selectedValue, selectedValue + loadingDuration};
      }
      currentIndex++;
    }
  }

  const auto lastValue = get<1>(selectedNodeBooking[iterLimit]);
  if(lastValue < arrivalHour)
    return {arrivalHour, arrivalHour + loadingDuration};
  return {lastValue, lastValue + loadingDuration};
}

void Graph::bookReservation(nodeId concernedNode,veId concernedVehicle,
    minutes beginTime, minutes endTime) const
{
  bookings[concernedNode].push_back({beginTime, endTime, concernedVehicle});
  sort(begin(bookings[concernedNode]), end(bookings[concernedNode]),
      [](const auto& a, const auto& b){ return get<0>(a) < get<0>(b); });
}

pair<vector<nodeId>,minutes> Graph::backtrackPath(nodeId origin, nodeId destination) const
{
  vector<nodeId> buildedPath;
  nodeId currentNode = destination;

  while(currentNode != origin)
  {
    buildedPath.push_back(currentNode);
    currentNode = parents[currentNode];
  }

  buildedPath.push_back(origin);
  reverse(begin(buildedPath), end(buildedPath));
  return {buildedPath, currentTime[destination]};
}

vector<nodeId> Graph::getCluster(nodeId center) const {
  vector<nodeId> cluster = {center};
  for(const auto& [neighbor, distance] : getSuccessors(center))
    if(distance < 50'000)
      cluster.push_back(neighbor);
  return cluster;
}

void Graph::generateRequests(GenMode mode, size_t numReqs, size_t numEVs,
                             meters minRange, meters maxRange,
                             minutes maxStartTime) const {
  TimeScope("Generating requests");

  uniform_int_distribution<meters> distrRange(minRange, maxRange);
  uniform_int_distribution<nodeId> distrNodes(0u, nodeCount - 1);
  uniform_int_distribution<minutes> distrTimes(0u, maxStartTime);

  for(size_t i = 0; i < numReqs; ++i) {
    newReq:
    vector<tuple<minutes, nodeId, nodeId, meters>> EVRequests;
    vector<nodeId> fromCluster;
    vector<nodeId> toCluster;
    uniform_int_distribution<nodeId> distrFrom;
    uniform_int_distribution<nodeId> distrTo;

    if(mode == GenMode::Uniform) {
      distrFrom = distrNodes;
      distrTo   = distrNodes;
    } else { // mode == GenMode::Cluster
      // Find two nodes at least 100km apart
      nodeId fromCenter = distrNodes(gen);
      nodeId toCenter   = distrNodes(gen);
      meters distance = getEdgeDistance(fromCenter, toCenter);
      while(distance < 100'000) {
        if(distance == 0)
          fromCenter = distrNodes(gen);
        toCenter = distrNodes(gen);
        distance = getEdgeDistance(fromCenter, toCenter);
      }

      // Find clusters
      fromCluster = getCluster(fromCenter);
      toCluster   = getCluster(toCenter);
      distrFrom   = uniform_int_distribution<nodeId>(0, size(fromCluster) - 1);
      distrTo     = uniform_int_distribution<nodeId>(0, size(toCluster) - 1);
    }

    size_t repCount = 0;
    for(size_t j = 0; j < numEVs; ++j) {
      const size_t startTime = distrTimes(gen);

      nodeId from, to;
      size_t rangeEV;
      while(true) {
        if(repCount++ > 100 * size(fromCluster) * size(toCluster))
          goto newReq;
        from = distrFrom(gen);
        to   = distrTo(gen);
        rangeEV = distrRange(gen);
        if(mode == GenMode::Cluster) {
          from = fromCluster[from];
          to   = toCluster[to];
        }

        if(getEdgeDistance(from, to) < 100'000) continue;
        const auto& [path, _1, _2]
          = AStar<TypeReserv::noReserve>(*this, Ve(1, from, to, rangeEV, 0));
        if(size(path) == 0) continue;
        break;
      }

      EVRequests.push_back({startTime, from, to, rangeEV});
    }

    sort(begin(EVRequests), end(EVRequests));
    for(const auto& [startTime, from, to, rangeEV] : EVRequests)
      cout << startTime << " " << from << " " << to << " " << rangeEV << "\n";
    cout << "===\n";
  }
}

vector<pair<veId,minutes>> Graph::fixNode(nodeId selectedNode) const
{
  vector<pair<veId, minutes>> modified;
  vector<tuple<minutes, minutes, veId>> finalBooking;

  if(size(bookings[selectedNode]) > 0)
  {
    // Adding first reservation
    finalBooking.push_back(bookings[selectedNode][0]);

    for(size_t i = 1; i < size(bookings[selectedNode]); ++i)
    {
      const auto& [arrivalTimeFinal, departureTimeFinal, idEVFinal] = finalBooking.back();
      const auto& [arrivalTimeCurr, departureTimeCurr, idEVCurr] = bookings[selectedNode][i];
      if(arrivalTimeCurr >= departureTimeFinal)
        finalBooking.push_back(bookings[selectedNode][i]);
      else
      {
        minutes difference = departureTimeFinal - arrivalTimeCurr;
        finalBooking.push_back({departureTimeFinal, departureTimeCurr + difference, idEVCurr});
        modified.push_back({idEVCurr, difference});
      }
    }

    bookings[selectedNode] = finalBooking;
  }
  return modified;
}

void Graph::driftBooking(nodeId consideredNode, veId consideredVe, minutes driftTime) const
{
  for(size_t i = 0; i < size(bookings[consideredNode]); ++i)
  {
    auto& [arrivalTime, departureTime, idEV] = bookings[consideredNode][i];
    if(idEV == consideredVe) {
      arrivalTime += driftTime;
      departureTime += driftTime;
      break;
    }
  }
}

pair<minutes,minutes> Graph::getBooking(nodeId selectedNode, veId selectedVe) const
{
  for(size_t i = 0; i < size(bookings[selectedNode]); ++i)
  {
    const auto& [arrivalTime, departureTime, idEV] = bookings[selectedNode][i];
    if(idEV == selectedVe)
      return {arrivalTime, departureTime};
  }

  return {0, 0};
}
