#include <queue>
#include <set>
#include <string>
#include <list>
#include <vector>
#include <map>

#include "algo.hpp"

using namespace std;

size_t nbVE = 0; // variable pour calculerN4
size_t nbBornes = 0;

class DateHeure {
  public:
    DateHeure(size_t nbsecondes_ = 0) : nbsecondes(nbsecondes_) {}
    DateHeure(size_t j, size_t h, size_t m = 0, size_t s = 0)
      : nbsecondes(60 * (60 * ((24 * j) + h) + m) + s) {}

    bool operator <(const DateHeure& dh) const {return nbsecondes < dh.nbsecondes;}
    bool operator <=(const DateHeure& dh) const {return nbsecondes <= dh.nbsecondes;}
    bool operator !=(const DateHeure& dh) const {return nbsecondes != dh.nbsecondes;}
    bool operator ==(const DateHeure& dh) const {return nbsecondes == dh.nbsecondes;}
    bool operator >(const DateHeure& dh) const {return nbsecondes > dh.nbsecondes;}
    bool operator >=(const DateHeure& dh) const {return nbsecondes >= dh.nbsecondes;}

    const DateHeure& operator +=(const DateHeure& dh) {
      nbsecondes += dh.nbsecondes;
      return *this;
    }

    DateHeure operator +(const DateHeure& dh) const {
      return DateHeure(nbsecondes + dh.nbsecondes);
    }

    const DateHeure& operator -=(const DateHeure& dh) {
      nbsecondes -= dh.nbsecondes;
      return *this;
    }

    DateHeure operator -(const DateHeure& dh) const {
      return DateHeure(nbsecondes - dh.nbsecondes);
    }

    const DateHeure& operator *=(size_t facteur) {
      nbsecondes *= facteur;
      return *this;
    }

    DateHeure operator *(size_t facteur) const {
      return DateHeure(nbsecondes * facteur);
    }

    size_t getEnSecondes() const {return nbsecondes;}

  private:
    size_t nbsecondes{};
};

struct EtatVE {
  nodeId borne{};
  DateHeure dh{};
};

struct Etat {
  Etat() : ves(nbVE), bornes(nbBornes) {}

  vector<EtatVE> ves{};
  vector<DateHeure> bornes{};
  DateHeure f{}, g{};
  size_t i{}, p{};
};

veId quelVEbouge(const Etat& e1, const Etat& e2) {
  for (veId i = 0; i < nbVE; i++)
    if (e1.ves[i].borne != e2.ves[i].borne)
      return i;
  return nbVE+1; // should never happen
}

void copier(const Etat& e1, Etat& e2) {
  for (veId i = 0; i < nbVE; i++)
    e2.ves[i] = e1.ves[i];
  for (nodeId i = 0; i < nbBornes; i++)
    e2.bornes[i] = e1.bornes[i];
  e2.f = e1.f;
  e2.g = e1.g;
  e2.i = e1.i;
  e2.p = e1.p;
}

struct CompareEtat {
  bool operator()(const Etat& e1, const Etat& e2) const {
    for (veId i = 0; i < nbVE; i++) {
      if (e1.ves[i].borne < e2.ves[i].borne)
        return true;
      if (e1.ves[i].borne > e2.ves[i].borne)
        return false;
      if (e1.ves[i].dh < e2.ves[i].dh)
        return true;
      if (e1.ves[i].dh > e2.ves[i].dh)
        return false;
    }
    return false;
  }
};

struct CompareNoyau {
  bool operator()(const Etat& e1, const Etat& e2) const {
    for (veId i = 0; i < nbVE; i++) {
      if (e1.ves[i].borne < e2.ves[i].borne)
        return true;
      if (e1.ves[i].borne > e2.ves[i].borne)
        return false;
    }
    return false;
  }
};

struct CompareF {
  bool operator()(const Etat& e1, const Etat& e2) const {
    return e1.f > e2.f;
  }
};

struct NoyauConteneur : public vector<Etat> {
  bool estDomine(const Etat& e) {
    //cout << "estDomine: e=" << e << "\n";
    for (const Etat& x : *this) {
      //cout << "x=" << x << "\n";
      bool domine = true;
      for (veId i = 0; i < nbVE; i++)
        if (e.ves[i].dh < x.ves[i].dh) {
          domine = false;
          break;
        }
      if (domine)
        return true;
    }
    return false;
  }
};

std::vector<unitarySolution>
calculerN4(const Graph& nodeGraph, const vector<Ve>& requests) {
  nbVE = size(requests);
  nbBornes = nodeGraph.getNodeCount();

  vector<Etat> etats;
  set<Etat, CompareEtat> open1;
  priority_queue<Etat, std::vector<Etat>, CompareF> open2;
  set<Etat, CompareEtat> close;
  map<Etat, NoyauConteneur, CompareNoyau> noyaux;

  // État initial
  Etat e0;
  e0.i = 0;
  e0.p = numeric_limits<size_t>::max();
  e0.f = e0.g = 0;
  for (veId i = 0; i < nbVE; i++) {
    e0.ves[i].borne = requests[i].getStart();
    e0.ves[i].dh = DateHeure(requests[i].getStartTime()*60);
    if (e0.ves[i].dh > e0.f)
      e0.f = e0.g = e0.ves[i].dh;
  }

  for (nodeId i = 0; i < nbBornes; i++)
    e0.bornes[i] = 0;
  etats.push_back(e0);
  open1.insert(e0);
  open2.push(e0);
  noyaux[e0].push_back(e0);

  // Etat travail
  Etat et;

  // Boucle principale
  while (!open2.empty()) {
    const Etat e = open2.top();
    open1.erase(e);
    close.insert(e);

    // But atteint ?
    bool butAtteint = true;
    for (veId i = 0; i < nbVE; i++)
      if (e.ves[i].borne != requests[i].getDest()) {
        butAtteint = false;
        break;
      }
    if (butAtteint)
      break;

    open2.pop();

    // Énumérer les déplacements possibles
    // Tous les VÉs peuvent bouger
    for (veId i = 0; i < nbVE; i++) {
      // Énumérer les arêtes sortantes
      const Ve& vehicle = requests[i];

      for (const auto& [successorId, distance] : nodeGraph.getSuccessors(e.ves[i].borne)) {
        if (vehicle.getAutonomy() < distance) // borne inatteignable
          continue;

        // Vérifier si ce VÉ est déjà allé à cette borne précédemment (plan depuis l'état initial).
        const Etat* ep = &e;
        bool veDejaEteBorne = false;
        while (ep != nullptr) {
          if (ep->ves[i].borne == successorId) {
            veDejaEteBorne = true;
            break;
          }
          ep = ep->p == numeric_limits<size_t>::max() ? nullptr : &(etats[ep->p]);
        }
        if (veDejaEteBorne)
          continue;

        // copie état
        copier(e, et);

        // faire circuler le VÉ
        auto [duree_route, reloadingTime] = requests[i].computeHopCost(distance);
        duree_route *= 60;
        reloadingTime *= 60;

        const DateHeure arrivee = e.ves[i].dh + duree_route;
        DateHeure debut = arrivee;
        if (successorId != requests[i].getDest() && e.bornes[successorId] > debut)
          debut = e.bornes[successorId];

        DateHeure fin = debut;
        if (successorId != requests[i].getDest())
        {
          fin = debut + reloadingTime;
          et.bornes[successorId] = fin;
        }

        et.ves[i].borne = successorId;
        et.ves[i].dh = fin;

        if (fin > et.g)
          et.g = fin;
        et.f = et.g;

        // Heuristique similaire à f = g + h
        for (veId v = 0; v < nbVE; v++) {
          DateHeure arriveMin = et.ves[v].dh;
          const double dist = static_cast<double>(nodeGraph.getHeuristicDistance(et.ves[v].borne,
                requests[v].getDest()));
          arriveMin += requests[v].computeTravelUnbounded(static_cast<meters>(dist));
          if (arriveMin > et.f)
            et.f = arriveMin;
        }

        // Déjà vu ?
        if (close.find(et) != close.end())
          continue;

        const auto& iopen1 = open1.find(et);
        if (iopen1 != open1.end() && iopen1->f < et.f)
          continue;

        const auto& inoyaux = noyaux.find(et);
        NoyauConteneur* nc = nullptr;
        if (inoyaux != noyaux.end())
        {
          nc = &(inoyaux->second);
          if (nc->estDomine(et))
            continue;
        }
        Etat temp;
        copier(et, temp);

        temp.i = etats.size();
        temp.p = e.i;
        etats.push_back(temp);
        open1.insert(temp);
        open2.push(temp);

        if (nc == nullptr)
          nc = &noyaux[temp];
        nc->push_back(temp);
      }
    }
  }

  // Extraction du plan
  vector<tuple<vector<nodeId>, minutes, meters>> results;
  if (!open2.empty()) {
    Etat e = open2.top();
    list<Etat> sequence;
    sequence.push_front(e);
    while (e.p != numeric_limits<size_t>::max()) {
      e = etats[e.p];
      sequence.push_front(e);
    }

    for (veId i = 0; i < nbVE; ++i)
      results.push_back({vector<nodeId>({requests[i].getStart()}), 0, 0});

    e = sequence.front();
    sequence.pop_front();

    for (const Etat& e2 : sequence) {
      const veId v = quelVEbouge(e, e2);
      auto& [currPlan, currMinutes, currMeters] = results[v];
      const meters distance = nodeGraph.getEdgeDistance(currPlan.back(), e2.ves[v].borne);
      const minutes travelTime = requests[v].computeTravelUnbounded(distance);

      currPlan.push_back(e2.ves[v].borne);
      currMeters += distance;

      if(e2.ves[v].borne == requests[v].getDest())
        currMinutes =
          static_cast<minutes>(static_cast<double>(e.ves[v].dh.getEnSecondes()) / 60.0)
          + travelTime - requests[v].getStartTime();
      e = e2;
    }
  }

  return results;
}
